unit IWCompListbox;
{PUBDIST}

interface

uses
  {$IFDEF VSNET}System.ComponentModel, System.Drawing, {$ENDIF}
  {$IFDEF VSNET}
  IWNetClasses,
  {$ELSE}
  Classes,
  {$IFDEF Linux}
  QTypes, IWCLXComponent, IWCLXClasses,
  {$ELSE}
  IWVCLComponent, IWVCLClasses,
  {$ENDIF}
  {$IFDEF Linux}QControls, {$ELSE}Controls, {$ENDIF}
  {$ENDIF}
  {$IFDEF Linux}QGraphics, {$ELSE}Graphics, {$ENDIF}
  IWHTMLTag, IWControl, IWScriptEvents, IWRenderContext, IWBaseInterfaces,
  IWColor, IWFileReference, IWGLobal;

type
  TIWCustomListCombo = class(TIWCustomControl)
  private
    FAutoHide: Boolean;
    FItems: TIWStringList;
  protected
    FBGColor: TIWColor;
    FItemIndex: Integer;
    FItemsHaveValues: Boolean;
    FRequireSelection: Boolean;
    FNoSelectionText: string;
    FOnChange: TNotifyEvent;
    FUseSize: Boolean;
    FRequired: Boolean;
    FFocusColor: TIWColor;

    procedure InitItems; virtual;
    procedure SetBGColor(const Value: TIWColor);
    procedure OnItemsChange(ASender: TObject);
    procedure DoChange;
    function GetSorted: boolean;
    function getText: TCaption; override;
    procedure HookEvents(AContext: TIWPageContext40; AScriptEvents: TIWScriptEvents); override;
    procedure SetItems(AValue: TIWStringList);
    function GetItems: TIWStringList;
    procedure SetItemIndex(AIndex: Integer); virtual;
    procedure SetRequireSelection(const AValue: Boolean); virtual;
    procedure SetSorted(AValue: boolean);
    procedure SetUseSize(const AValue: Boolean);
    procedure SetItemsHaveValues(const Value: Boolean);

    procedure RenderScripts(AComponentContext: TIWBaseHTMLComponentContext); override;

    procedure InitControl; override;
    function RenderOnChangeIntoTag(AUserAgent: string): Boolean; virtual;

    {$IFDEF VSNET}
    [DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
    {$ENDIF}
    property Items: TIWStringList read GetItems write SetItems;

    {$IFDEF VSNET}
    [Bindable(true)]
    {$ENDIF}    
    property ItemIndex: Integer read FItemIndex write SetItemIndex;

    // Do not publish sorted. DB controls use this and using it can interfere with their
    // functionality. Such as the lookuplist ones which rely on the sort order remaining the same
    // as fetched from the DB.
    //
    //The items in the TIWCustomListCombo are sorted.
    //
    //
    property Sorted: Boolean read GetSorted write SetSorted;
{$IFDEF CLR}
    strict protected
{$ELSE}
  protected
{$ENDIF}
    procedure Dispose(ADispose: Boolean); override;
  public
    function get_ShouldRenderTabOrder: boolean;override;
    procedure Clear; virtual;
    function RenderStyle(AContext: TIWBaseHTMLComponentContext): string; override;

    property Text: TCaption read getText write SetText;
  published
    property Enabled;
    property BGColor: TIWColor read FBGColor write SetBGColor;
    property ExtraTagParams;
    {$IFDEF CLR}
    property WebFont;
    {$ELSE}
    property Font;
    {$ENDIF}
    property FocusColor: TIWColor read FFocusColor write FFocusColor;
    property AutoHideOnMenuActivation: Boolean read FAutoHide write FAutoHide;
    property ItemsHaveValues: Boolean read FItemsHaveValues write SetItemsHaveValues;
    property NoSelectionText: string read FNoSelectionText write FNoSelectionText;
    property Required: Boolean read FRequired write FRequired;
    property RequireSelection: Boolean read FRequireSelection write SetRequireSelection;
    property ScriptEvents;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
    property UseSize: Boolean read FUseSize write SetUseSize;
  end;

  TIWCustomComboBoxStyle = (stNormal, stEnhanced);
  TIWCustomComboBox = class(TIWCustomListCombo, IIWInputControl, IIWSubmitControl)
  protected
    FSubmitParam : String;
    FStyle : TIWCustomComboBoxStyle;
    FButtonColor : TIWColor;
    FButtonImage : TIWFileReference;
    procedure InitControl; override;

    procedure SetValue(const AValue: string); virtual;
    procedure Submit(const AValue: string); override;
    function RenderOnChangeIntoTag(AUserAgent: string): Boolean; override;
  public
    function RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag; override;
    function GetSubmitParam : String;
    // Needed for PaintHandlers
    property Items;
    property ItemIndex;
    constructor Create{$IFNDEF VSNET}(AOwner : TComponent){$ENDIF}; override;
    destructor Destroy; override;
  published
    property Style : TIWCustomComboBoxStyle read FStyle write FStyle;
    property ButtonColor : TIWColor read FButtonColor write FButtonColor;
    property ButtonImage : TIWFileReference read FButtonIMage write FButtonImage;
    property DoSubmitValidation;
    property Editable;
    property NonEditableAsLabel;
    property TabOrder;
  end;

{$IFDEF VSNET}
{$R icons\Atozed.Intraweb.TIWComboBox.bmp}
  TIWComboBox = class;
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWComboBox), 'TIWComboBox.bmp')]
{$ENDIF}
  TIWComboBox = class(TIWCustomComboBox)
  published
    property ItemIndex;
    property Items;
    property Sorted;
    property FriendlyName;
  end;

  TIWCustomListbox = class(TIWCustomListCombo, IIWInputControl, IIWSubmitControl)
  protected
    FSubmitParam : String;
    FMultiSelect: Boolean;
    FSelectedList: TList;
    FTempList: TIWStringList;
    FOnDblClick: TNotifyEvent;
    //
{$IFNDEF VSNET}
    function GetOnClick: TNotifyEvent;
    procedure SetOnClick(const AValue: TNotifyEvent);
{$ENDIF}
    procedure DoItemsChange(Sender: TObject);
    procedure DoItemsChanging(Sender: TObject);
    function GetSelected(AIndex: integer): boolean;
    procedure SetSelected(AIndex: integer; const AValue: boolean);
    procedure SetMultiSelect(const AValue: Boolean);
    procedure HookEvents(AContext: TIWPageContext40; AScriptEvents: TIWScriptEvents); override;

    procedure SetValue(const AValue: string); virtual;
    procedure Submit(const AValue: string); override;

    procedure InitItems; override;
    procedure InitControl; override;
{$IFDEF CLR}
    strict protected
{$ELSE}
  protected
{$ENDIF}
    procedure Dispose(ADispose: Boolean); override;
  public
    function RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag; override;
    function GetSubmitParam : String;
    // Needed for PaintHandlers
    property Items;
    property ItemIndex;

    property MultiSelect: boolean read FMultiSelect write SetMultiSelect;

    procedure ResetSelection;

    property Selected[AIndex: Integer]: Boolean read GetSelected write SetSelected;
  published
{$IFNDEF VSNET}
    property OnClick: TNotifyEvent read GetOnClick write SetOnClick;
{$ENDIF}
    property OnDblClick: TNotifyEvent read FOnDblClick write FOnDblClick;
    property Confirmation;
    property DoSubmitValidation;
    property Editable;
    property TabOrder;
  end;

  {$IFDEF VSNET}
  {$R icons\Atozed.Intraweb.TIWListbox.bmp}
  TIWListbox = class;
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWListbox), 'TIWListbox.bmp')]
  {$ENDIF}
  TIWListbox = class(TIWCustomListbox)
  protected
    procedure SetRequireSelection(const AValue: Boolean); override;
  published
    property FriendlyName;
    property ItemIndex;
    property Items;
    property MultiSelect: boolean read FMultiSelect write SetMultiSelect;
    property Sorted;
  end;

implementation

uses
  Math,
  SysUtils,
{$IFDEF VSNET}
{$ELSE}
{$IFDEF Linux}
  QForms,
  Types,
{$ELSE}
  Forms,
  Windows,
{$ENDIF}
{$ENDIF}
  IWApplication, IWServerControllerBase, IWAppForm, IWCompLabel, IWTypes,
  IWResourceStrings, IWServer,

  SWStrings, SWSystem, IWBaseHTMLControl, IWMarkupLanguageTag, HTTPApp;

type
  TIWStringList2 = class(TIWStringList)
  protected
    FOwner:TIWCustomListCombo;
    procedure InsertItem(Index: Integer; const S: string; AObject: TObject); {$IFNDEF VER130}{$IFNDEF CBUILDER5}override;{$ENDIF}{$ENDIF}
    procedure QuickSort(L, R: Integer; SCompare: TStringListSortCompare);reintroduce;
  public
    constructor Create(const AOwner:TIWCustomListCombo);reintroduce;
    procedure Delete(Index: Integer); override;
    procedure Exchange(Index1, Index2: Integer); override;
    procedure CustomSort(Compare: TStringListSortCompare); override;
  end;

{ TIWCustomListCombo }


function TIWCustomListCombo.get_ShouldRenderTabOrder: boolean;
begin
  result := Editable or (NonEditableAsLabel = false);
end;

procedure TIWCustomListCombo.OnItemsChange(ASender: TObject);
begin
  DoRefreshControl := true;
  Invalidate;
end;

procedure TIWCustomListCombo.InitControl;
begin
  inherited;
  FBGColor := fromTColor(clNone);
  FNeedsFormTag := True;
  RequireSelection := True;
  FRequired := False;
  FNoSelectionText := '-- No Selection --';
  FItemIndex := -1;
  FCanReceiveFocus := True;
  FFocusColor := fromTColor(clNone);
  Enabled := True;
  FUseSize := True;
end;

procedure TIWCustomListCombo.Dispose(ADispose: Boolean);
begin
  FreeAndNil(FItems);
  inherited;
end;

function TIWCustomListCombo.GetSorted: boolean;
begin
  Result := TIWStringList(Items).Sorted;
end;

procedure TIWCustomListCombo.DoChange;
begin
  if Assigned(OnChange) then begin
    OnChange(Self);
  end;
end;

{ RenderOnChangeIntoTag - tells whether the OnChange JavaScript
  code should be put into SELECT tag (in other case HookEvent
  will be used.) Recuired for Mac IE compatibility. }

function TIWCustomListCombo.RenderOnChangeIntoTag(AUserAgent: string): Boolean;
begin
  result := false; { Returns false by default. This behavior
  will be changed in overriden RenderOnChangeIntoTag method
  of TIWCustomComboBox }
end;

procedure TIWCustomListCombo.HookEvents(AContext: TIWPageContext40; AScriptEvents: TIWScriptEvents);
begin
  inherited HookEvents(AContext, AScriptEvents);
  if Editable then begin
    if not RenderOnChangeIntoTag(AContext.WebApplication.Request.UserAgent) then begin
      AScriptEvents.HookEvent('OnChange', iif(Assigned(OnChange), SubmitHandler('')));
    end;
  end;

  // FocusColor support
  if (toTColor(FocusColor) <> clNone) and editable then begin
    AScriptEvents.HookEvent('OnFocus', 'this.style.backgroundColor=''' + ColorToRGBString(FocusColor) + ''';');
    AScriptEvents.HookEvent('OnBlur', 'this.style.backgroundColor=''' + ColorToRGBString(BGColor) + ''';');
  end;
end;

{ RenderOnChangeIntoTag - tells whether the OnChange JavaScript
  code should be put into SELECT tag (in other case HookEvent
  will be used.) Recuired for Mac IE compatibility. }

function TIWCustomComboBox.RenderOnChangeIntoTag(AUserAgent: string): Boolean;
begin
  Result := (Pos('MSIE', UpperCase(AUserAgent)) > 0)
    and ((Pos('MACINTOSH', UpperCase(AUserAgent)) > 0)
    or (Pos('MAC', Uppercase(AUserAgent)) > 0));
end;

constructor TIWCustomComboBox.Create{$IFNDEF VSNET}(AOwner : TComponent){$ENDIF};
begin
  inherited;
  FButtonImage := TIWFileReference.Create;
  FButtonColor := fromTColor(clBtnFace);
end;

destructor TIWCustomComboBox.Destroy;
begin
  FButtonImage.Free;
  inherited Destroy;
end;


function TIWCustomComboBox.GetSubmitParam: String;
begin
  Result := FSubmitParam;
end;

function TIWCustomComboBox.RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag;

  function RenderSelectedBoxStyle : string;
  var
    sWidth, sHeight, sClipWidth, sClipHeight : string;
  begin
    case AContext.Browser of
      brNetscape6,
      brNetscape7:  sWidth := IntToStr(Width-22);
      else          sWidth := IntToStr(Width-18);
    end;
    case AContext.Browser of
      brNetscape6,
      brNetscape7:  sClipWidth := IntToStr(Width-18);
      else          sClipWidth := sWidth;
    end;
    case AContext.Browser of
      brOpera:      sHeight := IntToStr(Height);
      brNetscape6,
      brNetscape7:  sHeight := IntToStr(Height-4);
      else          sHeight := IntToStr(Height+1);
    end;
    case AContext.Browser of
      brNetscape6,
      brNetscape7:  sClipHeight := IntToStr(Height+1);
      else          sClipHeight := sHeight;
    end;
    Result :=
      'position:absolute; cursor: pointer; border-style: inset; border-width:2px; z-index: 1;'+
      'top: 0px; left: 0px; width:'+sWidth+'px; height: '+sHeight+'px; '+
      WebFont.FontToStringStyle(AContext.Browser)+'; padding-left: 5; '+
      'background-color: '+ColorToRGBString(BGColor)+'; clip:rect(0px,'+sClipWidth+'px,'+sClipHeight+'px,0px);';
  end;

  function RenderSelectedTextStyle : string;
  var
    sWidth, sHeight : string;
  begin
    sWidth := IntToStr(Width-18);
    sHeight := IntToStr(WebFont.Size);
    Result := 'background-color:'+ColorToRGBString(BGColor)+'; height:'+IntToStr(WebFont.Size)+'px; width:'+sWidth+'px;clip:rect(0px,'+sWidth+'px,'+sHeight+'px,0px);';
  end;

  function RenderSelectedText : string;
  begin
    if ItemIndex < 0 then
      Result := NoSelectionText
    else if FItemsHaveValues then
      Result := Items.Names[ItemIndex]
    else
      Result := Items[ItemIndex];
  end;
                                          
  function RenderButtonStyle : string;
  var
    sWidth, sHeight : string;
  begin
    sWidth := IntToStr(Width-18);
    sHeight := IntToStr(Height);
    Result :=
      'position:absolute; cursor :pointer; border-width: 1px; border-style: outset;  z-index: 2;'+
      'top: 0px; left:'+sWidth+'px; width:18px;height:'+sHeight+'px; clip:rect(0px,22px,'+sHeight+'px,0px); '+
      'background-color: '+ColorToRGBString(FButtonColor);
  end;

  function RenderImageStyle : string;
  begin
    Result := 'position: absolute;left: -1px;top:1px;';
  end;

  function RenderImageSource : string;
  begin
    Result := FButtonImage.Location(GServerController.FilesURL);
    if Result = '' then
    begin
      Result := AContext.WebApplication.InternalURLBase + '/gfx/ComboBoxDownArrow.gif';
    end;
  end;

  function RenderDropdownStyle : string;
  var
    LWidth: string;
    LHeight: string;
    LColor: IWColor.TIWColor;
  begin
    LColor := BGCOlor;

    if LColor = {$IFDEF VSNET}System.Drawing.Color.Empty {$ELSE}clNone{$ENDIF} then
      LColor := {$IFDEF VSNET}System.Drawing.Color.White {$ELSE}clWebWhite{$ENDIF};

    case AContext.Browser of
      brNetscape6,
      brNetscape7:  LWidth := IntToStr(Width-8);
      else          LWidth := IntToStr(Width);
    end;

    LHeight := IntToStr(Height);
    Result :=
      'position:absolute; border-style:inset; padding-left:5; visibility: hidden; overflow:auto; border-width:2px;  z-index: 3;'+
      'top: ' + LHeight + 'px; left: 0px; width:'+LWidth+'px; height: ' + IntToStr(Min(95, (Items.Count + 2) * 14)) + 'px; clip:rect(0px,'+IntToStr(Width+1)+'px,100px,0px); '+
      'background-color:'+ColorToRGBString(LColor)+'; '+ WebFont.FontToStringStyle(AContext.Browser);
  end;

  function RenderItemStyle : string;
  {$IFNDEF VSNET}
  var
    LColor : TIWColor;
  begin
    LColor := BGCOlor;
    if LColor = fromTColor(clNone) then
      LColor := fromTColor(clWebWhite);
    Result := 'background-color:'+ColorToRGBString(LColor)+'; cursor :pointer;';
  {$ELSE}
  begin
    Result := '';
  {$ENDIF}
  end;

const
  CRLF = #13#10;
var
  i: Integer;
  LText: string;
  LValue: string;
  tag : TIWHTMLTag;
  LParam1: string;
  LParam2: string;
begin
  Result := nil;
  if Editable or (NonEditableAsLabel = false) then
  begin
    case FStyle of
    stNormal:
      begin
        Result := TIWHTMLTag.CreateTag('SELECT');
        try
          Result.AddStringParam('NAME', HTMLName);
          Result.AddIntegerParam('SIZE', 1);

          if not IsDesignMode and RenderOnChangeIntoTag(AContext.WebApplication.Request.UserAgent) then
          begin
            if Assigned(OnChange) then
            begin
              Result.AddStringParam('OnChange', SubmitHandler);
            end;
          end;

          if not Enabled then begin
            Result.Add('DISABLED');
          end;

           if (not Editable) and (NonEditableAsLabel = false) then begin
            Result.AddBoolParam('DISABLED', true);
            Result.AddBoolParam('READONLY', true);
          end;

          if FUseSize then
          begin
            Result.AddIntegerParam('WIDTH', Width);
          end;
          {Result.Add('STYLE', iif(WebApplication.Browser in [brIE, brNetscape6]
             , Font.FontToStringStyle(WebApplication.Browser)){ + 'width: ' + IntToStr(Width) + 'px;');}

          if Required then
          begin
            TIWComponent40Context(AContext).AddValidation(HTMLName + 'IWCL != null && ' + HTMLName + 'IWCL.value=="-1"'
              , Format(RSSelectionRequired, [FriendlyName]));
          end;

          if Items.Count > 0 then
          begin
            if ((FItemIndex = -1) or (RequireSelection = False)) then
            begin
              with Result.Contents.AddTag('OPTION') do
              begin
                if FItemIndex = -1 then begin
                  Add('SELECTED');
                end;
                AddIntegerParam('VALUE', -1);
                Contents.AddText(FNoSelectionText);
              end;
            end;
            for i := 0 to Items.Count - 1 do
            begin
              LText := Items.Strings[i];
              LValue := '';
              if ItemsHaveValues then
              begin
                LValue := LText;
                LText := Fetch(LValue, '=');
              end
              else
              begin
                LValue := IntToStr(i);
              end;
              with Result.Contents.AddTag('OPTION') do
              begin
                Add(iif(ItemIndex = i, 'SELECTED'));
                AddStringParam('VALUE', LValue);
                Contents.AddText(TextToHTML(LText));
              end;
            end;
          end
          else
          begin
            Result.Contents.AddText('');
          end;
        except
          FreeAndNil(Result);
          raise;
        end;
      end;
    stEnhanced:
      begin
        Result := TIWHTMLTag.CreateTag('SPAN');
        try
          // Render Hidden Input
          with Result.Contents.AddTag('INPUT') do
          begin
            AddStringParam('NAME',HTMLName);
            AddStringParam('type','hidden');
            AddStringParam('value',IntToStr(ItemIndex));
          end;

          // Render Javascript
          tag := Result.Contents.AddTag('SCRIPT');
//          tag.AddStringParam('language','JavaScript1.2');
          tag.AddStringParam('type','text/javascript');
          tag.Contents.AddText(
            '  var '+HTMLName+'_state = ''closed'';'+CRLF+
            '  function '+HTMLName+'_doClick() {'+CRLF);
          if Enabled then
            tag.Contents.AddText(
              '    if ('+HTMLName+'_state ==''closed'') {'+CRLF+
              '      document.getElementById('''+HTMLName+'_Button'').style.borderStyle = ''inset'';'+CRLF+
              '      document.getElementById('''+HTMLName+'_DropDown'').style.visibility = ''visible'';'+CRLF+
              '      '+HTMLName+'_state = ''open'';'+CRLF+
              '    } else {'+CRLF+
              '      document.getElementById('''+HTMLName+'_Button'').style.borderStyle = ''outset'';'+CRLF+
              '      document.getElementById('''+HTMLName+'_DropDown'').style.visibility = ''hidden'';'+CRLF+
              '      '+HTMLName+'_state = ''closed'';'+CRLF+
              '    }'+CRLF);
          tag.Contents.AddText(
            '  }'+CRLF+
            '  function '+HTMLName+'_doSelectItem(idx, text) {'+CRLF+
            '    '+HTMLName+'_doClick();'+CRLF+
            '    document.getElementById('''+HTMLName+'_Text'').innerHTML = text;'+CRLF+
            '    FindElem('''+HTMLName+''').value = idx;'+CRLF);
          if Assigned(OnChange) then
          begin
            tag.Contents.AddText(SubmitHandler('''+idx+''',HTMLName));
          end;
          tag.Contents.AddText('  }'+CRLF);

          // Render Selected Item Box
          tag := Result.Contents.AddTag('SPAN');
          tag.AddStringParam('ID',HTMLName+'_Selected');
          tag.AddStringParam('style',RenderSelectedBoxStyle);
          tag.AddStringParam('onClick',HTMLName+'_doClick();');

          with tag.Contents.AddTag('DIV') do
          begin
            AddStringParam('ID',HTMLName+'_Text');
            AddStringParam('style',RenderSelectedTextStyle);
            Contents.AddText(TextToHTML(RenderSelectedText));
          end;

          // Render Button
          tag := Result.Contents.AddTag('SPAN');
          tag.AddStringParam('ID',HTMLName+'_Button');
          tag.AddStringParam('style',RenderButtonStyle);
          tag.AddStringParam('onClick',HTMLName+'_doClick();');
          with tag.Contents.AddTag('IMG') do
          begin
            AddIntegerParam('width',17);
            AddIntegerParam('height',18);
            AddStringParam('style',RenderImageStyle);
            AddStringParam('src',RenderImageSource);
          end;

          // Render Dropdown
          tag := Result.Contents.AddTag('SPAN');
          tag.AddStringParam('ID',HTMLName+'_DropDown');
          tag.AddStringParam('Style',RenderDropdownStyle);

          if ((FItemIndex = -1) or (RequireSelection = False)) then
          begin
            with tag.Contents.AddTag('DIV') do
            begin
              AddStringParam('ID',HTMLName+'_item_noselect');
              AddStringParam('Style',RenderItemStyle);
              AddStringParam('onClick',HTMLName+'_doSelectItem(-1,'''+TExtToHTML(FNoSelectionText)+''');');
              Contents.AddTag('NOBR').Contents.AddText(TextToHTML(FNoSelectionText));
            end;
          end;
          for i := 0 to Items.Count-1 do
            with tag.Contents.AddTag('DIV') do
            begin
              AddStringParam('ID',HTMLName+'_item_'+IntToStr(i));
              AddStringParam('Style',RenderItemStyle);
              if FItemsHaveValues then begin
                LParam1 := Items.Values[Items.Names[i]];
                LParam2 := Items.Names[i];
                LParam1 := StringReplace(LParam1, '''', '\''', [rfReplaceAll]);
                LParam2 := StringReplace(LParam2, '''', '\''', [rfReplaceAll]);
                AddStringParam('onClick',HTMLName+'_doSelectItem('+LParam1+','''+LParam2+''');');
                Contents.AddTag('NOBR').Contents.AddText(TextToHTML(Items.Names[i]));
              end else begin
                LParam1 := StringReplace(Items[i], '''', '\''', [rfReplaceAll]);

                AddStringParam('onClick',HTMLName+'_doSelectItem('+IntToStr(i)+','''+LParam1+''');');
                Contents.AddTag('NOBR').Contents.AddText(TextToHTML(Items[i]));
              end;
            end;

          if Required then
          begin
            TIWComponent40Context(AContext).AddValidation(HTMLName + 'IWCL != null && ' + HTMLName + 'IWCL.value=="-1"'
              , Format(RSSelectionRequired, [FriendlyName]));
          end;
        except
          FreeAndNil(Result);
          raise;
        end;
      end;
    end;
  end else
  begin
    with TIWLabel.Create{$IFNDEF VSNET}(Self){$ENDIF} do
    try
      Name := Self.Name;
      BGColor := Self.BGColor;
      ExtraTagParams.Assign(Self.ExtraTagParams);
      WebFont.Assign(Self.WebFont);
      Width := Self.Width;
      Height := Self.Height;
      Caption := '';
      if (Self.Items.Count > 0) and (Self.ItemIndex > -1) then
      begin
        if Self.ItemsHaveValues then
        begin
          Caption := TextToHTML(Self.Items.Names[Self.ItemIndex]);
        end
        else
        begin
          Caption := TextToHTML(Self.Items.Strings[Self.ItemIndex]);
        end;
      end;
      Result := RenderHTML(AContext);
    finally
      Free;
    end;
  end;
end;

function TIWCustomListCombo.getText: TCaption;
begin
  if (ItemIndex > -1) and (ItemIndex < Items.Count) then begin
    if ItemsHaveValues then begin
      Result := Copy(Items[ItemIndex], AnsiPos('=', Items[ItemIndex]) + 1, Length(Items[ItemIndex]));
    end else begin
      Result := Items.Strings[ItemIndex];
    end;
  end else begin
    if ItemIndex >= Items.Count then begin
      ItemIndex := -1;
    end;
    Result := '';
  end;
end;

procedure TIWCustomListCombo.SetItemIndex(AIndex: Integer);
begin
  if IsLoading then begin
    // Set no matter what, it might be set (and usually is) before the items are loaded
    FItemIndex := AIndex;
  end
  else
  begin
    if AIndex < Items.Count then
    begin
      if FItemIndex <> AIndex then
      begin
        FItemIndex := AIndex;
        DoRefreshControl := True;
        Invalidate;
      end;
    end;
  end;
end;

procedure TIWCustomListCombo.SetItems(AValue: TIWStringList);
begin
  if (AValue.Count = 0) or (AValue.Count < Items.Count) then begin
    FItemIndex := -1;
  end;
  Items.Assign(AValue);
  Invalidate;
end;

procedure TIWCustomListCombo.SetRequireSelection(const AValue: Boolean);
begin
  FRequireSelection := AValue;
end;

procedure TIWCustomListCombo.SetSorted(AValue: boolean);
begin
  TIWStringList(Items).Sorted := AValue;
  Invalidate;
end;

procedure TIWCustomComboBox.SetValue(const AValue: string);
var
  s: string;
  i: integer;
begin
  if RequiresUpdateNotification(Parent) then begin
    UpdateNotifiedInterface(Parent).NotifyUpdate(Self,AValue);
  end;
  s := AValue;
  if ItemsHaveValues then begin
    for i := 0 to Items.Count - 1 do begin
      if S = Copy(Items[i], AnsiPos('=', Items[i]) + 1, Length(Items[i])) then begin
        if ItemIndex <> i then begin
          ItemIndex := i;
          Invalidate;
        end;
        break;
      end;
    end;
    if i = Items.Count then begin
      ItemIndex := -1;
      Invalidate;
    end;
  end else begin
    i := StrToIntDef(Fetch(s, ','), -1);
    if i <> ItemIndex then begin
      ItemIndex := i;
      Invalidate;
    end;
  end;
end;

procedure TIWCustomListCombo.SetUseSize(const AValue: Boolean);
begin
  Set_RenderSize(AValue);
  FUseSize := AValue;
end;

procedure TIWCustomListCombo.SetItemsHaveValues(const Value: Boolean);
begin
  FItemsHaveValues := Value;
  Invalidate;
end;

function TIWCustomListCombo.RenderStyle(
  AContext: TIWBaseHTMLComponentContext): string;
begin
  Result := inherited RenderStyle(AContext);
  if toTColor(BGColor) <> clNone then begin
    Result := Result + 'background-color:' + ColorToRGBString(BGColor) + ';';
  end;
end;

procedure TIWCustomListCombo.SetBGColor(const Value: TIWColor);
begin
  FBGColor := Value;
  Invalidate;
end;

procedure TIWCustomListCombo.RenderScripts(
  AComponentContext: TIWBaseHTMLComponentContext);
begin
  inherited;
  with TIWComponent40Context(AComponentContext) do
    AddToIWCLInitProc('if(' + HTMLName + 'IWCL) { '
      + HTMLName + 'IWCL.AutoHide = ' + iif(FAutoHide, 'true', 'false')
      + ' }');
end;

procedure TIWCustomListCombo.InitItems;
begin
  FItems := TIWStringList2.Create(Self);
  FItems.OnChange := OnItemsChange;
end;

function TIWCustomListCombo.GetItems: TIWStringList;
begin
  if not Assigned(FItems) then begin
    InitItems;
  end;
  result := FItems;
end;

{ TIWCustomListbox }

procedure TIWCustomListbox.InitControl;
begin
  inherited;
  FSelectedList := TList.Create;
  Height := 121;
  Width := 121;
  FTempList := TIWStringList.Create;
end;

procedure TIWCustomListbox.Dispose(ADispose: Boolean);
begin
  FreeAndNil(FTempList);
  FreeAndNil(FSelectedList);
  inherited;
end;

function TIWCustomListbox.GetSelected(AIndex: Integer): boolean;
begin
  if FMultiSelect then begin
    Result := FSelectedList.IndexOf(TObject(AIndex)) > -1;
  end else begin
    Result := AIndex = ItemIndex;
  end;
end;

function TIWCustomListbox.GetSubmitParam: String;
begin
  Result := FSubmitParam;
end;

function TIWCustomListbox.RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag;
var
  i: Integer;
  LText: string;
  LValue: string;
begin
  Result := nil;
  if Editable or (NonEditableAsLabel = false) then
  begin
    Result := TIWHTMLTag.CreateTag('SELECT');
    try
      Result.AddStringParam('NAME', HTMLName);
      Result.AddIntegerParam('SIZE', Height div 16);
      Result.Add(iif(FMultiSelect, 'MULTIPLE'));
      if FUseSize then
      begin
        Result.AddIntegerParam('WIDTH', Width);
      end;
        {Result.Add('STYLE', iif(WebApplication.Browser in [brIE, brNetscape6]
        , Font.FontToStringStyle(WebApplication.Browser)));}

      if not Enabled then begin
        Result.Add('DISABLED');
      end;

      if NonEditableAsLabel = false then begin
        Result.AddBoolParam('DISABLED', true);
        Result.AddBoolParam('READONLY', true);
      end;

      if Required then
      begin
        TIWComponent40Context(AContext).AddValidation(HTMLName + 'IWCL != null && ' + HTMLName + 'IWCL.value=="-1"'
          , Format(RSSelectionRequired, [FriendlyName]));
      end;

      if Items.Count > 0 then
      begin
        if ((FItemIndex = -1) or (RequireSelection = False))
          and (FMultiSelect = False) then
        begin
          with Result.Contents.AddTag('OPTION') do
          begin
            Add('SELECTED');
            AddIntegerParam('VALUE', -1);
            Contents.AddText(FNoSelectionText);
          end;
        end;
        for i := 0 to Items.Count - 1 do
        begin
          LText := Items.Strings[i];
          LValue := '';
          if ItemsHaveValues then
          begin
            LValue := LText;
            LText := Fetch(LValue, '=');
          end
          else
          begin
            LValue := IntToStr(i);
          end;
          with Result.Contents.AddTag('OPTION') do
          begin
            Add(iif((
              (ItemIndex = i) and (FMultiSelect = False))
              or (Selected[i] and FMultiSelect)
              , 'SELECTED'));
            AddStringParam('VALUE', LValue);
            Contents.AddText(TextToHTML(LText));
          end;
        end;
      end
      else
      begin
        Result.Contents.AddText('');
      end;
    except
      FreeAndNil(Result);
      raise;
    end;
  end
  else
  begin
    with TIWLabel.Create{$IFNDEF VSNET}(Self){$ENDIF} do
    try
      Name := Self.Name;
      BGColor := Self.BGColor;
      FFriendlyName := Self.FriendlyName;
      ExtraTagParams.Assign(Self.ExtraTagParams);
        //Font.Assign(Self.Font);
      ScriptEvents.Assign(Self.ScriptEvents);
      Width := Self.Width;
      Height := Self.Height;
      Caption := '';
      if Self.ItemIndex > -1 then
      begin
        if ItemsHaveValues then begin
          Caption := TextToHTML(Self.Items.Values[Self.Items.Names[Self.ItemIndex]]);
        end else begin
          Caption := TextToHTML(Self.Items.Strings[Self.ItemIndex]);
        end;
      end;
      Result := RenderHTML(AContext);
    finally
      Free;
    end;
  end;
end;

procedure TIWCustomListbox.ResetSelection;
begin
  FSelectedList.Clear;
  Invalidate;
end;

procedure TIWCustomListbox.SetSelected(AIndex: integer; const AValue: boolean);
begin
  if AValue then begin
    if not GetSelected(AIndex) then begin
      FSelectedList.Add(TObject(AIndex));
    end;
  end else begin
    FSelectedList.Remove(TObject(AIndex));
  end;
  Invalidate;
end;                                                                

procedure TIWCustomListbox.SetMultiSelect(const AValue: boolean);
begin
  if AValue then begin
    FRequireSelection := False;
  end;
  FMultiSelect := AValue;
end;

procedure TIWCustomListbox.SetValue(const AValue: string);
var
  LValue, s: string;
  i: Integer;
begin
  if RequiresUpdateNotification(Parent) then begin
    UpdateNotifiedInterface(Parent).NotifyUpdate(Self,AValue);
  end;
  LValue := AValue;
  s := Fetch(LValue, ',');
  if ItemsHaveValues then begin
    if AValue = '-1' then begin
      ItemIndex := -1;
    end else begin
      for i := 0 to Items.Count - 1 do begin
        if S = Copy(Items[i], AnsiPos('=', Items[i]) + 1, Length(Items[i])) then begin
          ItemIndex := i;
          break;
        end;
      end;
    end;
  end else begin
    ItemIndex := StrToIntDef(s, -1);
  end;
  // For multiselect, first one is ItemIndex. Will be repeated in selected list
  // ItemIndex := StrToIntDef(Fetch(LValue, ','), -1);
  if FMultiSelect then begin
    ResetSelection;
    Selected[ItemIndex] := true;
    while Length(LValue) > 0 do begin
      s := Fetch(LValue, ',');
      if ItemsHaveValues then begin
        for i := 0 to Items.Count - 1 do begin
          if S = Copy(Items[i], AnsiPos('=', Items[i]) + 1, Length(Items[i])) then
            break;
        end;
      end else begin
        i := StrToInt(s);
      end;

      Selected[i] := True;
    end;
  end;
end;

procedure TIWCustomListbox.Submit(const AValue: string);
begin
  FSubmitParam := AValue;
  DoChange;
  if AValue='DblClick' then begin
    if Assigned(OnDblClick) then begin
      OnDblClick(self);
    end;
  end else begin
    DoClick;
  end;
end;

procedure TIWCustomListbox.HookEvents(AContext: TIWPageContext40; AScriptEvents: TIWScriptEvents);
begin
  inherited HookEvents(AContext, AScriptEvents);
  if Editable then begin
    AScriptEvents.HookEvent('OnClick', iif(hasOnClick, SubmitHandler('')));
    AScriptEvents.HookEvent('OnDblClick', iif(Assigned(OnDblClick), SubmitHandler('DblClick')));
  end;
end;

procedure TIWCustomListbox.DoItemsChange(Sender: TObject);
var
  f: integer;
  LTempSelected: TList;
begin
(*
  if MultiSelect and (FSelectedList.Count > 0) then begin
    LTempSelected := TList.Create;
    try
      for f := 0 to Pred(FTempList.Count) do
      begin
        if Items.IndexOf(FTempList.Strings[f]) >= 0 then
        begin
          if FSelectedList.IndexOf(TObject(f)) >= 0 then
          begin
            LTempSelected.Add(TObject(Items.IndexOf(FTempList.Strings[f])));
          end;
        end;
      end;
      FSelectedList.Clear;
{$IFDEF VER130}
      FSelectedList.Capacity := LTempSelected.Count; //
      CopyMemory(FSelectedList.List, LTempSelected.List, SizeOf(LTempSelected.List^[0]) * LTempSelected.Count);
{$ELSE}
      FSelectedList.Assign(LTempSelected);
{$ENDIF}
    finally
      FreeAndNil(LTempSelected);
    end;
  end;
  OnItemsChange(Sender);
  *)
end;

procedure TIWCustomListbox.DoItemsChanging(Sender: TObject);
begin
  if not (MultiSelect and (FSelectedList.Count > 0)) then
  begin
    Exit;
  end;

  FTempList.Assign(Items);
end;

procedure TIWCustomListbox.InitItems;
begin
  inherited;
  FItems.OnChange := DoItemsChange;
  FItems.OnChanging := DoItemsChanging;
end;

{ TIWCustomComboBox }

procedure TIWCustomComboBox.InitControl;
begin
  inherited;
  Height := 21;
  Width := 121;
end;

procedure TIWCustomComboBox.Submit(const AValue: string);
begin
  FSubmitParam := AValue;
  DoChange;
end;

procedure TIWCustomListCombo.Clear;
begin
  ItemIndex := -1;
  Items.Clear;
end;

{ TIWListbox }

procedure TIWListbox.SetRequireSelection(const AValue: Boolean);
begin
  if AValue then begin
    FMultiSelect := False;
  end;
  inherited;
end;

{$IFNDEF VSNET}

function TIWCustomListBox.GetOnClick: TNotifyEvent;
begin
  Result := (inherited OnClick);
end;

procedure TIWCustomListBox.SetOnClick(const AValue: TNotifyEvent);
begin
  inherited OnClick := AValue;
end;
{$ENDIF}

{ TIWStringList2 }


procedure TIWStringList2.InsertItem(Index: Integer; const S: string; AObject: TObject);
var
  a:integer;
begin
  {$IFDEF VER130}
  AddObject(S, AObject);
  {$ELSE}
  {$IFDEF CBUILDER5}
  AddObject(S, AObject);
  {$ELSE}
  inherited InsertItem(Index, S, AObject);
  {$ENDIF}
  {$ENDIF}
  if FOwner is TIWCustomListbox then begin
    for a:=Count - 1 downto Index + 1 do begin
      TIWCustomListbox(FOwner).Selected[a] := TIWCustomListbox(FOwner).Selected[a-1];
    end;
    TIWCustomListbox(FOwner).Selected[Index]:=false;
  end;
end;

constructor TIWStringList2.Create(const AOwner:TIWCustomListCombo);
begin
  inherited Create;
  FOwner := AOwner;
end;

procedure TIWStringList2.Delete(Index: Integer);
var
  a:integer;
begin
  inherited Delete(Index);
  if FOwner is TIWCustomListbox then begin
    for a:=Index to Count do begin
      TIWCustomListbox(FOwner).Selected[a] := TIWCustomListbox(FOwner).Selected[a+1];
    end;
    TIWCustomListbox(FOwner).Selected[Count] := false;
  end;
end;

procedure TIWStringList2.Exchange(Index1, Index2: Integer);
begin
  inherited Exchange(Index1, Index2);
  if FOwner is TIWCustomListbox then begin
    if TIWCustomListbox(FOwner).Selected[Index1] and not TIWCustomListbox(FOwner).Selected[Index2] then begin
     TIWCustomListbox(FOwner).Selected[Index1] := false;
     TIWCustomListbox(FOwner).Selected[Index2] := true;
    end else if not TIWCustomListbox(FOwner).Selected[Index1] and TIWCustomListbox(FOwner).Selected[Index2] then begin
     TIWCustomListbox(FOwner).Selected[Index1] := true;
     TIWCustomListbox(FOwner).Selected[Index2] := false;
    end;
  end;
end;

procedure TIWStringList2.QuickSort(L, R: Integer; SCompare: TStringListSortCompare);
var
  I, J, P: Integer;
begin
  repeat
    I := L;
    J := R;
    P := (L + R) shr 1;
    repeat
      while SCompare(Self, I, P) < 0 do Inc(I);
      while SCompare(Self, J, P) > 0 do Dec(J);
      if I <= J then
      begin
        Exchange(I, J);
        if P = I then
          P := J
        else if P = J then
          P := I;
        Inc(I);
        Dec(J);
      end;
    until I > J;
    if L < J then QuickSort(L, J, SCompare);
    L := I;
  until I >= R;
end;

procedure TIWStringList2.CustomSort(Compare: TStringListSortCompare);
begin
  if not Sorted and (Count > 1) then
  begin
    Changing;
    QuickSort(0, Count - 1, Compare);
    Changed;
  end;
end;

initialization
  TIWServer.AddInternalFile('IW_GFX_ComboBoxDownArrow', '/gfx/ComboBoxDownArrow.gif');

end.

